
# wall_module_compiler.py
# FreeCAD compiler: Wall / Window / Door modules from schema → .FCStd

import os, sys, json, math, traceback

try:
    import FreeCAD as App
    import Part
except Exception as e:
    raise RuntimeError("Run inside FreeCAD or with freecadcmd") from e

INCH = 25.4

def mm(x): return float(x) * INCH
def v(x,y,z): return App.Vector(mm(x), mm(y), mm(z))

ACTUAL_2X_THICK = 1.5
ACTUAL_2X6_DEPTH = 5.5
ACTUAL_2X12_DEPTH = 11.25

def safe(name):
    return "".join(c if c.isalnum() or c=="_" else "_" for c in name)

def load_spec(path):
    with open(path, "r", encoding="utf-8") as f:
        txt = f.read()
    try:
        return json.loads(txt)
    except Exception:
        try:
            import yaml
            return yaml.safe_load(txt)
        except Exception:
            raise RuntimeError("Spec must be JSON or YAML")

def make_group(doc, name, parent=None):
    g = doc.addObject("App::DocumentObjectGroup", safe(name))
    if parent: parent.addObject(g)
    return g

def make_box(doc, name, lx, ly, lz, x, y, z, grp):
    b = doc.addObject("Part::Box", safe(name))
    b.Length = mm(lx)
    b.Width  = mm(ly)
    b.Height = mm(lz)
    b.Placement.Base = v(x,y,z)
    grp.addObject(b)
    return b

def stud_centers(width, oc):
    centers = [0.75]
    n = 1
    while 0.75 + n*oc < width-0.75:
        centers.append(0.75 + n*oc)
        n+=1
    centers.append(width-0.75)
    return centers

def compile_wall(spec, out_path):
    mod = spec["module"]
    W = mod["width_in"]
    H = mod["height_in"]
    oc = mod.get("stud_spacing_oc_in", 24)

    plates = mod.get("plates", {"top_plate_count":2,"bottom_plate_count":1})
    top_n = plates.get("top_plate_count",2)
    bot_n = plates.get("bottom_plate_count",1)

    openings = spec.get("openings", [])

    doc = App.newDocument(safe(mod.get("id","WallModule")))
    root = make_group(doc, "WallModule")

    framing = make_group(doc, "Framing", root)
    studs_g = make_group(doc, "Studs", framing)
    plates_g = make_group(doc, "Plates", framing)
    open_g = make_group(doc, "Openings", framing)

    stud_len = H - ACTUAL_2X_THICK*(top_n+bot_n)

    # Bottom plates
    for i in range(bot_n):
        make_box(doc, f"BottomPlate_{i}", W, ACTUAL_2X6_DEPTH,
                 ACTUAL_2X_THICK, 0, 0, i*ACTUAL_2X_THICK, plates_g)

    # Top plates
    for i in range(top_n):
        make_box(doc, f"TopPlate_{i}", W, ACTUAL_2X6_DEPTH,
                 ACTUAL_2X_THICK, 0, 0,
                 H-ACTUAL_2X_THICK*(top_n-i), plates_g)

    # Regular studs
    for i,c in enumerate(stud_centers(W, oc)):
        make_box(doc, f"Stud_{i}", ACTUAL_2X_THICK,
                 ACTUAL_2X6_DEPTH, stud_len,
                 c-ACTUAL_2X_THICK/2, 0,
                 ACTUAL_2X_THICK*bot_n, studs_g)

    # Openings
    for o in openings:
        og = make_group(doc, o.get("id","Opening"), open_g)
        ro = o["rough_opening"]
        w = ro["width_in"]
        h = ro["height_in"]
        x0 = o.get("x_from_left_in",
                   (W-w)/2 if o.get("centered_in_module") else 0)

        z0 = o.get("sill_height_from_bottom_in",0)
        z1 = z0 + h

        # Kings
        make_box(doc,"King_L",ACTUAL_2X_THICK,ACTUAL_2X6_DEPTH,stud_len,
                 x0-ACTUAL_2X_THICK*2,0,ACTUAL_2X_THICK*bot_n,og)
        make_box(doc,"King_R",ACTUAL_2X_THICK,ACTUAL_2X6_DEPTH,stud_len,
                 x0+w+ACTUAL_2X_THICK,0,ACTUAL_2X_THICK*bot_n,og)

        # Jacks
        jack_len = z1-ACTUAL_2X_THICK*bot_n
        make_box(doc,"Jack_L",ACTUAL_2X_THICK,ACTUAL_2X6_DEPTH,jack_len,
                 x0-ACTUAL_2X_THICK,0,ACTUAL_2X_THICK*bot_n,og)
        make_box(doc,"Jack_R",ACTUAL_2X_THICK,ACTUAL_2X6_DEPTH,jack_len,
                 x0+w,0,ACTUAL_2X_THICK*bot_n,og)

        # Header
        make_box(doc,"Header_2x12x2",
                 w+ACTUAL_2X_THICK*2,
                 ACTUAL_2X_THICK*2,
                 ACTUAL_2X12_DEPTH,
                 x0-ACTUAL_2X_THICK,0,z1+ACTUAL_2X_THICK,og)

    doc.recompute()
    doc.saveAs(out_path)

def main(argv):
    if len(argv)<3:
        print("Usage: freecadcmd wall_module_compiler.py spec.json out.fcstd")
        return
    spec = load_spec(argv[1])
    compile_wall(spec, argv[2])
    print("Wrote", argv[2])

if __name__=="__main__":
    main(sys.argv)
